(* ::Package:: *)

BeginPackage["BumpWave6`"]
EPS::usage="10^(-5) constant used in calculations"

makeSampleTable::usage="Given (t,gammaWidth,numPerGamma,kappa,f) builds a table arr of length numPerGamma+1.  arr[[i]] stores the max of f(t,s1,s2) over all s1,s2 values with max(|s1|,|s2|)<=i*gammaWidth+EPS.
We only consider intervals for s1,s2 of distance at least kappa-2*gammaWidth+EPS.  We will also use this function when gamma is the uniform grid with for the sparse noise proof.  In that
case a bound on all smaller grid widths is a 'crude' upper bound, but not that crude as we expect smaller grid widths to be 'better'.  arr[[i]] not meaningful for 2*i*gammaWidth <= kappa, as there will be no interval pairs satisfying the constraints.
Do not run with kappa-2*gammaWidth-EPS <= 0, as this will negate the effect of kappa and usually leads to bad results."
precisionMax::usage="Given (f,t,gInt,factor), computes the max of f(t,gInt), where gInt is an Interval object, by dividing gInt into factor subintervals for greater precision"
makeSampleTableSparse::usage="Given (t,gammaWidth,numPerGamma,f,factor) builds a table arr of length numPerGamma+1. arr[[i]] stores the max of f(t,s) where s lies in the interval [(i-1)*gammaWidth-EPS,i*gammaWidth+EPS].
This is called where f is the dampened noise function and its derivatives, and s represents the uniform mesh width.  Uses precisionMax with given factor to compute maxes to higher precision.  arr[[1]] stores the value Infinity."
makeFunctionTable::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,f,eps,tail) builds a table arr of dimension {numDelta*numPerDelta+1,numPerGamma+1}.
arr[[i,j]] stores an upper bound on f(t,s1,s2) for t in the interval [(i-1)*DeltaWidth-EPS,i*DeltaWidth+EPS] and max(|s1|,|s2|)<=i*gammaWidth+EPS while satisfying d(s1,s2) >= kappa+EPS.  If (i-1)*DeltaWidth >= tail+EPS, the value eps is used.
arr[[i,j]] not meaningful for (j-1)*gammaWidth < kappa-EPS."
makeFunctionTableSparse::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,f,eps,tail,factor) builds a table arr of dimension {numDelta*numPerDelta+1,numPerGamma+1}.
arr[[i,j]] stores an upper bound on f(t,s)) for t in the interval [(i-1)*DeltaWidth-EPS,i*DeltaWidth+EPS] and s in the interval [(j-1)*gammaWidth,j*gammaWidth].  If (i-1)*DeltaWidth >= tail+EPS, the value eps is used.
arr[[i,j]] not meaningful for j==1."

pairwiseMaxEps::usage="Given (a,b,eps) where a,b are lists of the same length and eps is a number.  Produces a list c of the same length where c[[i]] = max(a[[i]],b[[i]],eps).  Used to monotonize a function table."
makeMonotone::usage="Given (tab,eps) where tab is a function table and eps is a number.  The resulting table arr is monotonized so that arr[[i,j]] >= max(arr[[k,j]],eps) for any k>=j.  Used to obtain monotonic upper bounds on non-negative functions."

makeFun::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,f,eps,tail) calls makeFunctionTable"
makeAbsFun::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,f,eps,tail) calls makeFunctionTable using |f| and then calls makeMonotone on the result using eps"
makeAbsFunSparse::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,f,eps,tail,factor) calls makeFunctionTable using |f| and then calls makeMonotone on the result using eps"

computeBumpWave::usage="Given (K) or (K,DK) algebraically computes the bump, wave, and dampened noise functions.  Returns the list of functions: (B,W,S,dB,dW,dS,ddB,ddW,ddS)."

makeFunctionsBW::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,bwlist,eps,tail,factor) makes a list of function tables for bumps, waves, and dampened noise using makeFun, makeAbsFun, and makeAbsFunSparse.  
Make sure kappa-2*gammaWidth-EPS>0. Expects bwlist to be of the form output by computeBumpWave.  Returns the following list of functions where d denotes differentiation and A denotes absolute-monotonized: (AB,AW,AS,DB,ADB,ADW,ADS,DDB,ADDB,ADDW,ADDS)"
makeFunctions::usage="Given (gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,K,eps,tail,factor), calls makeFunctionsBW with computeBumpWave[K]"


computeNormMat::usage="Given (funList,DeltaIdx,numDelta,gammaIdx,eps) computes infinity norm bounds on each block in {{I-B,W},{DB,I-DW}} and returns them in a 2x2 matrix.  This is the identity minus the bump-wave block interpolation matrix.
Here we have Delta(T) = DeltaIdx*DeltaWidth and gamma(S,T) = gammaIdx*gammaWidth.  The sums are computed to numDelta terms, and have a tail approximation bound of eps."

computeAlphaBetaSparse::usage="Given (funList,DeltaIdx,numDelta,gammaIdx,AS,ADS,eps) computes (alpha,beta,alpha-eta,alphaLowerBound).  The first three are infinity norm bounds, the last is a lower bound on all entries of alpha.
Here we have Delta(T) = DeltaIdx*DeltaWidth and gamma(S,T) = gammaIdx*gammaWidth = upper bound on uniform mesh width for computing the matrix entries.  AS and ADS represent an upper bounds on the contribution of R_C,DR_C, the sum of the dampened noise
and its derivative, at a spike assumed to be at the origin. Returns failed if the interpolation matrix cannot be proven invertible."
computeAlphaBeta::usage="Given (funList,DeltaIdx,numDelta,gammaIdx,eps) computes (alpha,beta,alpha-rho,alphaLowerBound) by passing AS=0,ADS=0 to computeAlphaBetaSparse."

computeBounds::usage="Given (funList,DeltaIdx,numDelta,gammaIdx,eps) computes the list of arrays (AB,ANB,AW,DB,ADW,ADDB,DDB,ADDW) each of length Ceiling[DeltaIdx/2].
Assume, as we do in the paper, that there is a spike at the origin, and that t lies in the interval [0,Delta/2].
AB[[i]] stores an upper bound on the absolute sum of the bumps from all spikes to the range of t values in the interval [(i-1)*DeltaWidth,i*DeltaWidth].  DB, DDB have the same format but aren't absolute values.
ANB is a bound assuming the spike at the origin has value 0 (for dense noise calculation).  The bounds hold for Delta(T) >= DeltaIdx*DeltaWidth and gamma(S,T) <= gammaIdx*gammaWidth.  Returns $Failed if computeAlphaBeta fails."
computeBoundsDampened::usage="Given (funList,gammaIdx,DeltaIdx,numDelta,lambda,eps) computes the list of values (AS,ADS,ADDS).  AS upper bounds the value of |R_C(t)|, sum of the absolute dampened sparse noises, for all t.
Here each corrupted sample is separated by Delta >= DeltaIdx*DeltaWidth and has uniform mesh width gamma in [(gammaIdx-1)*gammaWidth,gammaIdx*gammaWidth]."
computeQBoundSparse::usage="Given (funList,DeltaMinGammaIdx,numDelta,gammaIdx,eps,AS,alpha,beta).  Let si and s(i+1) denote 2 samples surrounding a spike at the origin.  This function computes an upper bound
on the value of |Q(si)| using only the terms not involving si and s(i+1).  The closest distance from si to another sample is lower bounded by DeltaMinGammaIdx*DeltaWidth."
computeBoundsSparse::usage="Given (funList,DeltaIdx,DeltaMinGammaIdx,numDelta,gammaIdx,gammaWidth,lambda,eps,isGaussian) computes the list of arrays (AB,AW,AS,DB,ADW,ADS,ADDB,DDB,ADDW,ADS) where 
each array has length Ceiling[DeltaIdx/2]. Returns $Failed if system is not proved invertible, or if q cannot be bounded (uses Gaussian q bound if isGaussian, and Ricker otherwise).
Values are tested and results are computed where Delta(T) >= DeltaIdx*DeltaWidth and has uniform mesh width gamma in [(gammaIdx-1)*gammaWidth,gammaIdx*gammaWidth].  Delta(T) is also the corruption separation amount."

testQRegionBound::usage="Given (Q,DQ,DDQ) tests for overlapping regions of DDQ<0, DQ<0 and |Q|<1 to prove that |Q(t)|<1 for t in T^c.  Guarantees that Q is strongly concave near the spike at origin with sign +1. 
Assumes the first location where DDQ and DQ are both negative is the t1 value, so this function may be stricter than is necessary."
isSuccess::usage="Given (BoundList) computed by computeBounds tests whether |Q| is bounded and whether Q is strongly concave near the spike at the origin with sign +1."
isSuccessNoise::usage="Given (BoundList) computed by computeBounds tests whether |Q|<1 at a spike with value 0 (for local dense noise combiation) and that DDQ is never infinite (always true)."
isSuccessSparse::usage="Given (BoundList) computed by computeBoundsSparse tests whether |Q| is bounded and whether Q is strongly concave near the spike at the origin with sign +1."

makeKernelSuccessTable::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,eps) returns a table arr of dimensions {Length[gammaList],Length[DeltaList]}.
arr[[i,j]] stores whether exact recovery is proven for Delta(T) >= DeltaList[[j]], gamma(S,T) <= gammaList[[i]]."
makeKernelSuccessTableNoise::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,eps)  returns a table arr of dimensions {Length[gammaList],Length[DeltaList]}.
arr[[i,j]] stores whether exact recovery is proven for Delta(T) >= DeltaList[[j]], gamma(S,T) <= gammaList[[i]] by checking the exact recovery conditions and dense noise bounds."
makeKernelSuccessTableSparse::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,lambda,eps,isGaussian) returns a table arr of dimensions {Length[gammaList],Length[DeltaList]}.
arr[[i,j]] stores whether sparse recovery works for Delta(T) >= DeltaList[[j]] and the uniform grid width gamma in [(gammaIdx-1)*gammaWidth,gammaIdx*gammaWidth].  Unlike exact recovery and dense noise, proving sparse recovery for a fixed
grid width does not prove recovery for all smaller grid widths.  As a result, if gammaList is not a contiguous region of grid cells (separated by gammaWidth) then there will be 'holes' in the table."

makeGammaIdxList::usage="Given (gammaList,gammaWidth), rounds gammaList to the indices determined by a grid of width gammaWidth"
makeDeltaIdxList::usage="Given (DeltaList,DeltaWidth), rounds DeltaList to the indices determined by a grid of width DeltaWidth"

plotTable::usage="Given (table,gammaList,DeltaList,gammaWidth,DeltaWidth,K) plots the boolean table using the lists to get the ticks.  Ticks are rounded to the grid."
plotValueTable::usage="Given (table,gammaList,DeltaList,gammaWidth,DeltaWidth,K) plots the table using the lists to get the ticks.  Ticks are rounded to the grid."

plotBoundList::usage="Given (BoundList) constructed by computeBounds, plots Q,DQ,DDQ"
plotBoundListSparse::usage="Given (BoundListSparse) constructed by computeBoundsSparse, plots Q, DQ, DDQ"

makeIDWandICTables::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,eps) returns a list of 2 tables (IDW, IC).  IDW is a table of I-DW infinity norm bounds indexed gamma and Delta.  IC is the
same, but of I-C infinity norm bounds."
makeAlphaBetaTables::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,eps) returns a list of 4 tables (alpha,beta,alphaRho,alphaLB).  The tables contain the values from computeAlphaBeta indexed by gamma and Delta.
If the call fails, the returned entries are Infinity for the first 3 values and -Infinity for alphaLB."
makeQData::usage="Given (BoundList) computed by computeBounds, returns a list of three arrays (Q,DQ,DDQ) that can be used to plot these bounds."
computeT1Idx::usage="Given (BoundList) computed by computeBounds, returns the first location where DQ and DDQ are both negative.  The returned idx corresponds ot the interval [DeltaWidth*(idx-1),DeltaWidth*idx].  
Should be called when isSuccess[BoundList] is true."
makeT1Table::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,eps), computes a table where each cell is the computeT1Idx value.  Returns a negative value if isSuccess fails."

makeC2PTable::usage="Given (funList,gammaList,gammaWidth,DeltaList,DeltaWidth,numDelta,eps), computes a table of C2' values, that is, upper bounds on |Q| when using the local dual combination for dense noise. 
Returns a negative value if computeBounds fails."

computeSparseValues::usage="Given (funList,gammaLow,gammaHigh,gammaWidth,Delta,DeltaWidth,numDelta,lambda,eps,isGaussian) returns a list
(eta,zeta,alpha,beta,alphaEta,alphaj,Qsi,trueGammaLow,trueGammaHigh,trueDelta) of values needed for the sparse noise proof.  They are computed over the interval of uniform mesh widths [trueGammaLow,trueGammaHigh] for Delta(T) >= trueDelta.
Returns $Failed if the system is not proven invertible at any gamma value in the range."
pairwiseMax::usage="Given (a,b) computes their max if numeric, or their pairwise max if lists"
makeMaxSparseBoundList::usage="Given (funList,gammaLow,gammaHigh,gammaWidth,Delta,DeltaWidth,numDelta,lambda,eps,isGaussian) computes the max of results from computeBoundsSparse from all gammas between gammaLow and gammaHigh.
Returns $Failed if computeBoundsSparse fails at any gamma value in the range."
makeSparseQData::usage="Given (BoundListSparse) returns a list (Q,DQ,DDQ,AS,ADS,ADDS).  The first 3 are lists, the last 3 are values."


Begin["`Private`"]


(*Constant*)
EPS = 10^(-5);


(*Routines to build rows of the function tables*)
makeSampleTable[t_,gammaWidth_,numPerGamma_,kappa_,f_] :=
	Module[{ret,w,c,d,idx,v,lLeft,kLeft,lRight,kRight},
		w = gammaWidth;
		ret = Table[-Infinity,{i,0,numPerGamma}]; (*Result array*)
		Do[
			{lLeft,lRight} = {l*w-EPS,(l+1)*w+EPS}; (*Interval of s1 values *)
			{kLeft,kRight} = {k*w-EPS,(k+1)*w+EPS}; (*Interval of s2 values *)
			c = kappa - 2*w; (*Used to guarantee that kappa constraint is met*)
			d = Max[k*w-(l+1)*w,0]; (*Distance between s1 interval and s2 interval as k\[GreaterEqual]l*)
			v = Max[f[t,Interval[{lLeft,lRight}],Interval[{kLeft,kRight}]]]; (*Max of f over t, s1, s2 intervals*)
			idx = Max[-l,k+1]; (*The furthest index from the origin is the larger of -l and k+1 since k\[GreaterEqual]l*)
			ret[[idx]] = Max[ret[[idx]], If[d >= c-EPS, v, ret[[idx]]]]; (*Update value extra often to ensure kappa condition is met, i.e., that our upper bound is high enough*)
			,{l,-numPerGamma,numPerGamma}
			,{k,l,numPerGamma}
		];
		FoldList[Max,ret] (*Make monotone since gamma is an upper bound on sample separation*)
	]
precisionMax[f_,t_,gInt_,factor_] := 
	Module[{gL,gR,gW},
		{gL,gR} = {Min[gInt],Max[gInt]};
		gW = (gR-gL)/factor; (*Divide gInt into factor subintervals of width gW*)
		Max[Table[Max[f[t,Interval[{gL+k*gW-EPS,gL+(k+1)*gW+EPS}]]],{k,0,factor-1}]]
	]
makeSampleTableSparse[t_,gammaWidth_,numPerGamma_,f_,factor_] :=
	Module[{w},
		w = gammaWidth;
		Table[ If[g===0,Infinity,precisionMax[f,t,Interval[{g*w,(g+1)*w}],factor]], {g,0,numPerGamma} ]
	] 


(*Routines to build the function tables*)
makeFunctionTable[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,kappa_,f_,eps_,tail_] :=
	Module[{w},
		w = DeltaWidth;
		Print["Making Function Table"];
		ParallelTable[
			Module[ {DLeft, DRight},
				{DLeft,DRight} = {Max[w*t-EPS,0],w*(t+1)+EPS};
				If[DLeft>=tail,Table[eps,{i,0,numPerGamma}],makeSampleTable[Interval[{DLeft,DRight}],gammaWidth,numPerGamma,kappa,f]]
			]
			,{t,0,numDelta*numPerDelta}
			,DistributedContexts -> Automatic
		]
	]
makeFunctionTableSparse[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,f_,eps_,tail_,factor_] :=
	Module[{w},
		w = DeltaWidth;
		Print["Making Sparse Function Table"];
		ParallelTable[
			Module[ {DLeft, DRight},
				{DLeft,DRight} = {Max[w*t-EPS,0],w*(t+1)+EPS};
				If[DLeft>=tail,Table[eps,{i,0,numPerGamma}],makeSampleTableSparse[Interval[{DLeft,DRight}],gammaWidth,numPerGamma,f,factor]]
			]
			,{t,0,numDelta*numPerDelta}
			,DistributedContexts -> Automatic
		]
	]


(*Routines to monotonize a function table*)
pairwiseMaxEps[a_,b_,eps_] := MapThread[Max[#1,#2,eps]&,{a,b}] (*Used to take pairwise maxes of lists*)
makeMonotone[tab_,eps_] := Reverse[ FoldList[pairwiseMaxEps[#1,#2,eps]&, Reverse[tab]] ] (* folds the list in reverse using pairwiseMaxEps *)


(*Routines to build regular, or absolute monotonized function tables*)
makeFun[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,kappa_,f_,eps_,tail_] := makeFunctionTable[ gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,f,eps,tail] 
makeAbsFun[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,kappa_,f_,eps_,tail_] := makeMonotone[makeFunctionTable[ gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa, Abs[f[#1,#2,#3]]&,eps,tail],eps]
makeAbsFunSparse[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,f_,eps_,tail_,factor_] := makeMonotone[makeFunctionTableSparse[ gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta, Abs[f[#1,#2]]&,eps,tail,factor],eps]


(*Routines to algebraically compute and simplify the bump, wave, and dampened sparse noise functions centered at the origin*)
computeBumpWave[K_,DK_]:= 
	Module[{Minv,B,W,dB,dW,ddB,ddW,S,dS,ddS},
		Minv[t_,L_,R_] = Inverse[{{K[L],K[R]},{-DK[L],-DK[R]}}];
		Print["Simplifying Bump"];
		B[t_,L_,R_] = FullSimplify[({{K[L-t],K[R-t]}}.Minv[t,L,R].{{1},{0}})[[1,1]]];
		Print["Simplifying Wave"];
		W[t_,L_,R_] = FullSimplify[({{K[L-t],K[R-t]}}.Minv[t,L,R].{{0},{1}})[[1,1]]];
		Print["Simplifying Spike"];
		S[t_,g_] = FullSimplify[K[-t]-K[0]*B[t,-g,g]];
		Print["Simplifying DBump"];
		dB[t_,L_,R_] = FullSimplify[D[B[t,L,R],t]];
		Print["Simplifying DWave"];
		dW[t_,L_,R_] = FullSimplify[D[W[t,L,R],t]];
		Print["Simplifying DSpike"];
		dS[t_,g_] = FullSimplify[D[S[t,g],t]];
		Print["Simplifying DDBump"];
		ddB[t_,L_,R_] = FullSimplify[D[B[t,L,R],{t,2}]];
		Print["Simplifying DDWave"];
		ddW[t_,L_,R_] = FullSimplify[D[W[t,L,R],{t,2}]];
		Print["Simplifying DDSpike"];
		ddS[t_,g_] = FullSimplify[D[S[t,g],{t,2}]];
		{B,W,S,dB,dW,dS,ddB,ddW,ddS}
	]
computeBumpWave[K_]:= computeBumpWave[K,K']


(*Routines to make a list of function tables for the bumps, waves, and dampened noise for a given kernel*)
makeFunctionsBW[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,kappa_,bwlist_,eps_,tail_,factor_] :=
	Module[{B,W,S,dB,dW,dS,ddB,ddW,ddS},
		{B,W,S,dB,dW,dS,ddB,ddW,ddS} = bwlist;
		{
			makeAbsFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,B,eps,tail],
			makeAbsFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,W,eps,tail],
			makeAbsFunSparse[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,S,eps,tail,factor],
			makeFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,dB,eps,tail],
			makeAbsFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,dB,eps,tail],
			makeAbsFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,dW,eps,tail],
			makeAbsFunSparse[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,dS,eps,tail,factor],
			makeFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,ddB,eps,tail],
			makeAbsFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,ddB,eps,tail],
			makeAbsFun[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,ddW,eps,tail],
			makeAbsFunSparse[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,ddS,eps,tail,factor]
		} (*AB,AW,AS,DB,ADB,ADW,ADS,DDB,ADDB,ADDW,ADDS*)
	]
makeFunctions[gammaWidth_,DeltaWidth_,numPerGamma_,numDelta_,numPerDelta_,kappa_,K_,eps_,tail_,factor_] := makeFunctionsBW[gammaWidth,DeltaWidth,numPerGamma,numDelta,numPerDelta,kappa,computeBumpWave[K],eps,tail,factor]


(*Computes infinity norm upper bounds on the entries of the bump-wave interpolation matrix*)
computeNormMat[funList_,DeltaIdx_,numDelta_,gammaIdx_,eps_] :=
	Module[{ab,aw,adb,adw,IB,DB,W,IDW},
		ab =funList[[1]];
		aw =funList[[2]];
		adb =funList[[5]];
		adw =funList[[6]];
		IB = Sum[2*ab[[1+k*DeltaIdx,gammaIdx]],{k,1,numDelta}]+2*eps; (*Here ab[[1+k*DeltaIdx,gammaIdx]] upper bounds |B(t,s1,s2)| where t \[GreaterEqual] k*DeltaIdx*DeltaWidth and |s1|,|s2|\[LessEqual] gammaIdx*gammaWidth satisfying kappa constraint*)
		DB = Sum[2*adb[[1+k*DeltaIdx,gammaIdx]],{k,1,numDelta}]+2*eps; (*Sums are trimmed using tail bounds on the bumps, waves, and derivatives*)
		W = Sum[2*aw[[1+k*DeltaIdx,gammaIdx]],{k,1,numDelta}]+2*eps;
		IDW = Sum[2*adw[[1+k*DeltaIdx,gammaIdx]],{k,1,numDelta}]+2*eps;
		{{IB,W},{DB,IDW}}
	]


(*Routines to compute alpha, beta bounds for the exact and sparse noise cases*)
computeAlphaBetaSparse[funList_,DeltaIdx_,numDelta_,gammaIdx_,AS_,ADS_,eps_] :=
	Module[{IB,W,DB,IDW,DWInv,CInv,IC,alpha,beta,alphaeta},
		{{IB,W},{DB,IDW}} = computeNormMat[funList,DeltaIdx,numDelta,gammaIdx,eps];
		If[IDW >= 1-EPS,Return[$Failed],Null];
		DWInv = 1/(1-IDW); (*Infinity norm bound on inverse of DW*)
		IC = IB + W*DWInv*DB; (*Infinity norm bound on I-C where C is the Schur complement of DW*)
		If[IC >= 1-EPS,Return[$Failed],Null];
		CInv = 1/(1-IC); (*Infinity norm bound on inverse of C*)
		alpha = CInv*((1+AS)+W*DWInv*ADS);
		beta = DWInv*(ADS+DB*alpha);
		alphaeta = IC*alpha + W*DWInv*ADS;
		{alpha,beta,alphaeta,1-AS-alphaeta}
	](*alpha,beta,alpha-eta,alphaLB*)
computeAlphaBeta[funList_,DeltaIdx_,numDelta_,gammaIdx_,eps_] := computeAlphaBetaSparse[funList,DeltaIdx,numDelta,gammaIdx,0,0,eps] (*alpha,beta,alpha-rho,alphaLB*)


(*Routines for computing the bounds on Q, DQ, DDQ over the interval [0,Delta/2]*)
computeBounds[funList_,DeltaIdx_,numDelta_,gammaIdx_,eps_] :=
	Module[{coeffs,alpha,beta,alphaRho,alphaLB,ab,aw,as,db,adb,adw,ads,ddb,addb,addw,adds,AB,ANB,AW,DB,ADW,ADDB,DDB,ADDW}, 
		coeffs = computeAlphaBeta[funList,DeltaIdx,numDelta,gammaIdx,eps];
		If[coeffs===$Failed,Return[$Failed],{alpha,beta,alphaRho,alphaLB}=coeffs];
		{ab,aw,as,db,adb,adw,ads,ddb,addb,addw,adds} = funList;
		(*Note ab[[i+k*DeltaIdx,gammaIdx]] bounds |B(t,s1,s2)| for t \[GreaterEqual] (i-1+k*DeltaIdx)*DeltaWidth and |s1|,|s2| \[LessEqual] gammaIdx*gammaWidth *)
		(*Note db[[i+k*DeltaIdx,gammaIdx]] bounds DB(t,s1,s2) for t in [(i-1+k*DeltaIdx)*DeltaWidth,(i+k*DeltaIdx)*DeltaWidth] and |s1|,|s2| \[LessEqual] gammaIdx*gammaWidth *)
		AB = alpha*Table[ab[[i,gammaIdx]] + Sum[ab[[i+k*DeltaIdx,gammaIdx]]+ab[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}]; 
		ANB = Table[alphaRho*ab[[i,gammaIdx]] + alpha*(Sum[ab[[i+k*DeltaIdx,gammaIdx]]+ab[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps),{i,1,Ceiling[DeltaIdx/2]}];
		AW = beta*Table[aw[[i,gammaIdx]] + Sum[aw[[i+k*DeltaIdx,gammaIdx]]+aw[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		DB = Table[alphaLB*db[[i,gammaIdx]] + alpha*(Sum[adb[[i+k*DeltaIdx,gammaIdx]]+adb[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps),{i,1,Ceiling[DeltaIdx/2]}];
		ADW = beta*Table[adw[[i,gammaIdx]] + Sum[adw[[i+k*DeltaIdx,gammaIdx]]+adw[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		ADDB = alpha*Table[addb[[i,gammaIdx]] + Sum[addb[[i+k*DeltaIdx,gammaIdx]]+addb[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		DDB = Table[alphaLB*ddb[[i,gammaIdx]] + alpha*(Sum[addb[[i+k*DeltaIdx,gammaIdx]]+addb[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps),{i,1,Ceiling[DeltaIdx/2]}];
		ADDW = beta*Table[addw[[i,gammaIdx]] + Sum[addw[[i+k*DeltaIdx,gammaIdx]]+addw[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		{AB,ANB,AW,DB,ADW,ADDB,DDB,ADDW}
	]
computeBoundsDampened[funList_,gammaIdx_,DeltaIdx_,numDelta_,lambda_,eps_] :=
	Module[{coeffs,as,ads,adds,AS,ADS,ADDS},
		as =funList[[3]];
		ads =funList[[7]];
		adds =funList[[11]];
		AS = lambda*(2*Sum[as[[1+k*DeltaIdx,gammaIdx]],{k,0,numDelta}]+2*eps); (*as[[1+k*DeltaIdx,gammaIdx]] bounds |S(t,s)| for t\[GreaterEqual] k*DeltaIdx*DeltaWidth and s in [(gammaIdx-1)*gammaWidth,gammaIdx*gammaWidth] *)
		ADS = lambda*(2*Sum[ads[[1+k*DeltaIdx,gammaIdx]],{k,0,numDelta}]+2*eps);
		ADDS = lambda*(2*Sum[adds[[1+k*DeltaIdx,gammaIdx]],{k,0,numDelta}]+2*eps);
		{AS,ADS,ADDS}
	]
computeQBoundSparse[funList_,DeltaMinGammaIdx_,numDelta_,gammaIdx_,eps_,AS_,alpha_,beta_] :=
	Module[{B,W,DB,DW},
		{{B,W},{DB,DW}} = computeNormMat[funList,DeltaMinGammaIdx,numDelta,gammaIdx,eps];
		AS+alpha*B+beta*W
	]
computeBoundsSparse[funList_,DeltaIdx_,DeltaMinGammaIdx_,numDelta_,gammaIdx_,gammaWidth_,lambda_,eps_,isGaussian_] :=
	Module[{coeffs,alpha,beta,alphaEta,alphaLB,qBound,ab,aw,as,db,adb,adw,ads,ddb,addb,addw,adds,AB,AW,AS,DB,ADW,ADS,ADDB,DDB,ADDW,ADDS,betaBotFac,botGamma,topGamma},
		{botGamma,topGamma} = {(gammaIdx-1)*gammaWidth-EPS,gammaIdx*gammaWidth+EPS};
		{AS,ADS,ADDS} = computeBoundsDampened[funList,gammaIdx,DeltaIdx,numDelta,lambda,eps];
		coeffs = computeAlphaBetaSparse[funList,DeltaIdx,numDelta,gammaIdx,AS,ADS,eps];
		If[coeffs===$Failed,Return[$Failed],{alpha,beta,alphaEta,alphaLB}=coeffs];
		qBound = computeQBoundSparse[funList,DeltaMinGammaIdx,numDelta,gammaIdx,eps,AS,alpha,beta];
		betaBotFac = If[isGaussian,1,3-botGamma^2];
		If[lambda -beta/botGamma/betaBotFac- qBound <= 1+EPS,Return[$Failed],Null]; (*Verifies that Q(si)>1 if qi \[GreaterEqual] lambda*)
		If[beta*Exp[topGamma^2/2]/botGamma/betaBotFac >= lambda-EPS,Return[$Failed],Null]; (*Verifies that |q(i+1)| < lambda*)
		If[1-AS-alphaEta < 0+EPS,Return[$Failed],Null]; (*Verifies alpha_j \[GreaterEqual] 0 at a positive signed spike*)
		{ab,aw,as,db,adb,adw,ads,ddb,addb,addw,adds} = funList;
		AB = alpha*Table[ab[[i,gammaIdx]] + Sum[ab[[i+k*DeltaIdx,gammaIdx]]+ab[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		AW = beta*Table[aw[[i,gammaIdx]] + Sum[aw[[i+k*DeltaIdx,gammaIdx]]+aw[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		DB = Table[alphaLB*db[[i,gammaIdx]] + alpha*(Sum[adb[[i+k*DeltaIdx,gammaIdx]]+adb[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps),{i,1,Ceiling[DeltaIdx/2]}];
		ADW = beta*Table[adw[[i,gammaIdx]] + Sum[adw[[i+k*DeltaIdx,gammaIdx]]+adw[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		ADDB = alpha*Table[addb[[i,gammaIdx]] + Sum[addb[[i+k*DeltaIdx,gammaIdx]]+addb[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		DDB = Table[alphaLB*ddb[[i,gammaIdx]] + alpha*(Sum[addb[[i+k*DeltaIdx,gammaIdx]]+addb[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps),{i,1,Ceiling[DeltaIdx/2]}];
		ADDW = beta*Table[addw[[i,gammaIdx]] + Sum[addw[[i+k*DeltaIdx,gammaIdx]]+addw[[k*DeltaIdx-i,gammaIdx]],{k,1,numDelta}]+2*eps,{i,1,Ceiling[DeltaIdx/2]}];
		{AB,AW,AS,DB,ADW,ADS,ADDB,DDB,ADDW,ADS}
	]


(*Routines to test whether the dual combination Q is properly bounded*)
testQRegionBound[Q_,DQ_,DDQ_] :=
	Module[{isDQNeg,isDDQNeg,isQBound,i},
		isQBound = False;
		isDQNeg = False;
		isDDQNeg = True;
		For[ i = 1, i <= Length[Q], i++,
			If[isDDQNeg && DDQ[[i]] < -EPS, isDDQNeg = True, isDDQNeg = False];
			If[(isDDQNeg || isDQNeg) && DQ[[i]] < EPS, isDQNeg = True, isDQNeg = False];
			If[(isDDQNeg || isDQNeg || isQBound) && Q[[i]] <= 1-EPS,isQBound=True,isQBound=False];
			If[!isDDQNeg && !isDQNeg && !isQBound, Return[False], Null];
		];
		isQBound
	]
isSuccess[BoundList_] :=
	Module[{Q,DQ,DDQ},
		Q = BoundList[[1]]+BoundList[[3]];
		DQ = BoundList[[4]]+BoundList[[5]];
		DDQ = BoundList[[7]]+BoundList[[8]];
		testQRegionBound[Q,DQ,DDQ]
	]
isSuccessNoise[BoundList_] :=
	Module[{ANQ,ADDQ,i},
		ANQ = BoundList[[2]]+BoundList[[3]];
		ADDQ = BoundList[[6]]+BoundList[[8]];
		For[ i = 1, i <= Length[ANQ], i++,
			If[ANQ[[i]] > 1-EPS,Return[False],Null];
			If[ADDQ[[i]]===Infinity || ADDQ[[i]] === ComplexInfinity,Return[False],Null] (*Can never fail*)
		];
		Return[True]
	]
isSuccessSparse[BoundListSparse_] :=
	Module[{Q,DQ,DDQ,i},
		Q = BoundListSparse[[1]]+BoundListSparse[[2]]+BoundListSparse[[3]];
		DQ = BoundListSparse[[4]]+BoundListSparse[[5]]+BoundListSparse[[6]];
		DDQ = BoundListSparse[[8]]+BoundListSparse[[9]]+BoundListSparse[[10]];
		testQRegionBound[Q,DQ,DDQ]
	]


(*Rounds lists of values to the given grids*)
makeGammaIdxList[gammaList_,gammaWidth_] := Ceiling[gammaList/gammaWidth-EPS] (*Find the nearest grid point that is above each value since gamma is an upper bound*)
makeDeltaIdxList[DeltaList_,DeltaWidth_] := Floor[DeltaList/DeltaWidth+EPS] (*Find the nearest grid point that is below each value since Delta is a lower bound*)


(*Routines to make tables of whether proof is successful*)
makeKernelSuccessTable[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,eps_] :=
	Module[{DList,gList},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		ParallelTable[
			Module[{BoundList},
				BoundList = computeBounds[funList,DList[[j]],numDelta,gList[[i]],eps]; 
				If[BoundList===$Failed,False,isSuccess[BoundList]]
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
			,DistributedContexts -> Automatic
		]
	]
makeKernelSuccessTableNoise[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,eps_] :=
	Module[{EPS,DList,gList,BoundList},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		ParallelTable[
			Module[{QList},
				BoundList = computeBounds[funList,DList[[j]],numDelta,gList[[i]],eps]; 
				If[BoundList===$Failed,False,isSuccess[BoundList]&&isSuccessNoise[BoundList]]
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
			,DistributedContexts -> Automatic
		]
	]
makeKernelSuccessTableSparse[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,lambda_,eps_,isGaussian_] :=
	Module[{DList,gList,BoundListSpike},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		ParallelTable[
			Module[{DeltaIdx,gammaIdx,DeltaMinGammaIdx,topGamma,BoundListSparse},
				{DeltaIdx,gammaIdx} = {DList[[j]],gList[[i]]};
				topGamma = {(gammaIdx-1)*gammaWidth,gammaIdx*gammaWidth}; (*Upper bound on gamma interval*)
				DeltaMinGammaIdx = Floor[ (DeltaIdx*DeltaWidth - topGamma)/DeltaWidth + EPS ];
				BoundListSparse = computeBoundsSparse[funList,DeltaIdx,DeltaMinGammaIdx,numDelta,gammaIdx,gammaWidth,lambda,eps,isGaussian];
				If[BoundListSparse===$Failed,False,isSuccessSparse[BoundListSparse]]
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
			,DistributedContexts -> Automatic
		]
	]


(*Routines to plot tables of booleans or values with given tick lists*)
plotTable[table_,gammaList_,DeltaList_,gammaWidth_,DeltaWidth_,K_] :=
	Module[{m,xTicks,yTicks,gList,DList,t},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth]*gammaWidth,makeDeltaIdxList[DeltaList,DeltaWidth]*DeltaWidth};
		yTicks = Table[{k,DList[[k]]},{k,1,Length[DList]}];
		xTicks = Table[{k,Rotate[Text[gList[[k]]],Pi/2]},{k,1,Length[gList]}];
		m = MatrixPlot[Table[If[table[[j,i]],1,0],{i,1,Length[DList]},{j,1,Length[gList]}],FrameTicks->{yTicks,xTicks},Mesh->All];
		Show[m,AxesLabel->{None,None},FrameLabel->{{"\[CapitalDelta] (in units of \[Sigma])",None},{"\[Gamma] (in units of \[Sigma])",None}},PlotLabel->"K[t]="TraditionalForm[K[Global`t]],LabelStyle->{GrayLevel[0]}]
	]
plotValueTable[table_,gammaList_,DeltaList_,gammaWidth_,DeltaWidth_,K_] :=
	Module[{m,xTicks,yTicks,gList,DList,t},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth]*gammaWidth,makeDeltaIdxList[DeltaList,DeltaWidth]*DeltaWidth};
		yTicks = Table[{k,DList[[k]]},{k,1,Length[DList]}];
		xTicks = Table[{k,Rotate[Text[gList[[k]]],Pi/2]},{k,1,Length[gList]}];
		m = MatrixPlot[Table[table[[j,i]],{i,1,Length[DList]},{j,1,Length[gList]}],FrameTicks->{yTicks,xTicks},Mesh->All,PlotLegends -> Placed[BarLegend[Automatic], {After, Center}]];
		Show[m,AxesLabel->{None,None},FrameLabel->{{"\[CapitalDelta] (in units of \[Sigma])",None},{"\[Gamma] (in units of \[Sigma])",None}},PlotLabel->"K[t]="TraditionalForm[K[Global`t]],LabelStyle->{GrayLevel[0]}]
	]


(*Routines to plot the triple Q,DQ,DDQ*)
plotBoundList[BoundList_] :=
	ListLinePlot[{BoundList[[1]]+BoundList[[3]],BoundList[[4]]+BoundList[[5]],BoundList[[7]]+BoundList[[8]]},PlotRange->All,PlotLegends->{"Q","DQ","DDQ"}]
plotBoundListSparse[BoundListSparse_] := 
	ListLinePlot[{BoundListSparse[[1]]+BoundListSparse[[2]]+BoundListSparse[[3]],BoundListSparse[[4]]+BoundListSparse[[5]]+BoundListSparse[[6]],BoundListSparse[[8]]+BoundListSparse[[9]]+BoundListSparse[[10]]},PlotRange->All,PlotLegends->{"Q","DQ","DDQ"}]


(*Output for Exact Recovery*)
makeIDWandICTables[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,eps_] :=
	Module[{DList,gList,tab,IDWVals,ICVals},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		tab = Table[
			Module[{IB,W,DB,IDW,IC,DWInv},
				{{IB,W},{DB,IDW}} = computeNormMat[funList,DList[[j]],numDelta,gList[[i]],eps];
				DWInv = 1/(1-IDW);
				IC = IB + W*DWInv*DB;
				{IDW,IC}
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
		];
		IDWVals = Table[tab[[i,j,1]],{i,1,Length[gList]},{j,1,Length[DList]}];
		ICVals = Table[tab[[i,j,2]],{i,1,Length[gList]},{j,1,Length[DList]}];
		{IDWVals,ICVals,(gList*gammaWidth*1.0)[[1;;Length[gList]-1]],(DList*DeltaWidth*1.0)[[1;;Length[DList]-1]]}
	]
makeAlphaBetaTables[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,eps_] :=
	Module[{DList,gList,tab,alphaVals,betaVals,alphaRhoVals,alphaLBVals},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		tab = Table[
			Module[{v},
				v = computeAlphaBeta[funList,DList[[j]],numDelta,gList[[i]],eps];
				If[v === $Failed,{Infinity,Infinity,Infinity,-Infinity},v]
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
		];
		alphaVals = Table[tab[[i,j,1]],{i,1,Length[gList]},{j,1,Length[DList]}];
		betaVals = Table[tab[[i,j,2]],{i,1,Length[gList]},{j,1,Length[DList]}];
		alphaRhoVals = Table[tab[[i,j,3]],{i,1,Length[gList]},{j,1,Length[DList]}];
		alphaLBVals = Table[tab[[i,j,4]],{i,1,Length[gList]},{j,1,Length[DList]}];
		{alphaVals,betaVals,alphaRhoVals,alphaLBVals}
	]
makeQData[BoundList_]:= {BoundList[[1]]+BoundList[[3]],BoundList[[4]]+BoundList[[5]],BoundList[[7]]+BoundList[[8]]}
computeT1Idx[BoundList_] :=
	Module[{Q,DQ,DDQ,isDerNeg,isFunBound,i},
		{Q,DQ,DDQ} = makeQData[BoundList];
		For[ i = 1, i <= Length[Q], i++,
			If[DQ[[i]] < 0 && DDQ[[i]] < -EPS,Return[i],Null]
		];
		-1
	]
makeT1Table[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,eps_] :=
	Module[{DList,gList,tab,IDWVals,ICVals},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		ParallelTable[
			Module[{BoundList,isSucc},
				BoundList = computeBounds[funList,DList[[j]],numDelta,gList[[i]],eps]; 
				isSucc = If[BoundList===$Failed,False,isSuccess[BoundList]];
				If[isSucc,computeT1Idx[BoundList]*DeltaWidth,-100]
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
			,DistributedContexts -> Automatic
		]
	]


(*Output for Dense Noise*)
makeC2PTable[funList_,gammaList_,gammaWidth_,DeltaList_,DeltaWidth_,numDelta_,eps_] :=
	Module[{DList,gList,tab,IDWVals,ICVals},
		{gList,DList} = {makeGammaIdxList[gammaList,gammaWidth],makeDeltaIdxList[DeltaList,DeltaWidth]};
		ParallelTable[
			Module[{BoundList,val},
				BoundList = computeBounds[funList,DList[[j]],numDelta,gList[[i]],eps]; 
				If[BoundList===$Failed,-100,Max[ BoundList[[2]]+BoundList[[3]] ] ]
			]
			,{i,1,Length[gList]}
			,{j,1,Length[DList]}
			,DistributedContexts -> Automatic
		]
	]


(*Output for Sparse Noise*)
computeSparseValues[funList_,gammaLow_,gammaHigh_,gammaWidth_,Delta_,DeltaWidth_,numDelta_,lambda_,eps_,isGaussian_] :=
	Module[{gammaLowIdx,gammaHighIdx,DeltaIdx,eta,zeta,alpha,beta,alphaRho,alphaLB,alphaj,Qsi,i},
		gammaLowIdx = Ceiling[gammaLow/gammaWidth-EPS];
		gammaHighIdx = Ceiling[gammaHigh/gammaWidth-EPS];
		DeltaIdx = Floor[Delta/DeltaWidth+EPS];
		{eta,zeta,alpha,beta,alphaRho,alphaj,Qsi}={0,0,0,0,0,10^10,10^10};
		For[i = gammaLowIdx, i <= gammaHighIdx, i++,
			Module[{AS,ADS,ADDS,coeffs,alphaT,betaT,alphaRhoT,alphaLBT,qBound,betaFac,QsiT,DeltaMinGammaIdx,botGamma,topGamma},
				{AS,ADS,ADDS} = computeBoundsDampened[funList,i,DeltaIdx,numDelta,lambda,eps];
				coeffs = computeAlphaBetaSparse[funList,DeltaIdx,numDelta,i,AS,ADS,eps];
				If[coeffs===$Failed,Return[$Failed],{alphaT,betaT,alphaRhoT,alphaLBT}=coeffs];
				alpha = Max[alpha,alphaT];
				beta = Max[beta,betaT];
				alphaRho = Max[alphaRho,alphaRhoT];
				eta = Max[eta,1+AS];
				zeta = Max[zeta,AS];
				alphaj = Min[alphaj,1-AS-alphaRhoT];
				botGamma = (i-1)*gammaWidth;
				topGamma = i*gammaWidth;
				DeltaMinGammaIdx = Floor[(DeltaIdx*DeltaWidth-topGamma)/DeltaWidth+EPS];
				qBound = computeQBoundSparse[funList,DeltaMinGammaIdx,numDelta,i,eps,AS,alpha,beta];
				betaFac = If[isGaussian,1,3-botGamma^2];
				QsiT = lambda -beta/botGamma/betaFac- qBound;
				Qsi = Min[Qsi,QsiT];
			]
		];
		{eta,zeta,alpha,beta,alphaRho,alphaj,Qsi,(gammaLowIdx-1)*gammaWidth,gammaHighIdx*gammaWidth,DeltaIdx*DeltaWidth}
	]
pairwiseMax[a_,b_] := If[NumericQ[a],Max[a,b],MapThread[Max[#1,#2]&,{a,b}]]
makeMaxSparseBoundList[funList_,gammaLow_,gammaHigh_,gammaWidth_,Delta_,DeltaWidth_,numDelta_,lambda_,eps_,isGaussian_] :=
	Module[{BoundListSparse,gammaLowIdx,gammaHighIdx,DeltaIdx,i},
		gammaLowIdx = Ceiling[gammaLow/gammaWidth-EPS];
		gammaHighIdx = Ceiling[gammaHigh/gammaWidth-EPS];
		DeltaIdx = Floor[Delta/DeltaWidth+EPS];
		For[i = gammaLowIdx, i <= gammaHighIdx, i++,
			Module[{DeltaMinGammaIdx,botGamma,topGamma,newBounds},
				botGamma = (i-1)*gammaWidth;
				topGamma = i*gammaWidth;
				DeltaMinGammaIdx = Floor[(DeltaIdx*DeltaWidth-topGamma)/DeltaWidth+EPS];
				newBounds = computeBoundsSparse[funList,DeltaIdx,DeltaMinGammaIdx,numDelta,i,gammaWidth,lambda,eps,isGaussian];
				If[newBounds===$Failed,Print[{botGamma,topGamma,DeltaIdx*DeltaWidth,DeltaMinGammaIdx*Delta}],Null];
				If [i===gammaLowIdx,BoundListSparse = newBounds,BoundListSparse = MapThread[pairwiseMax[#1,#2]&,{BoundListSparse,newBounds}]];
			]
		];
		BoundListSparse
	]
makeSparseQData[BoundListSparse_] :={
	BoundListSparse[[1]]+BoundListSparse[[2]]+BoundListSparse[[3]],
	BoundListSparse[[4]]+BoundListSparse[[5]]+BoundListSparse[[6]],
	BoundListSparse[[8]]+BoundListSparse[[9]]+BoundListSparse[[10]],
	BoundListSparse[[3]],
	BoundListSparse[[6]],
	BoundListSparse[[10]]
}


End[]
EndPackage[]
