% Script to compare different algorithms to perform spectral 
% super-resolution. The data is of the form
% 
% y = Fx + z
%
% Where Fx corresponds to n samples of a multisinusoidal signal, x to the
% amplitudes of the spectrum of the signal and z to a sparse perturbation.
%
% Traditional methods fail in the presence of outliers, whereas an 
% approach based on convex programming, which models the sparse
% perturbation explicitly, is capable of detecting the outliers and 
% super-resolve the spectrum of the multisinusoidal component.
%
% For more information see the paper "Demixing Sines and Spikes: Spectral 
% Super-resolution in the Presence of Outliers" by C. Fernandez-Granda,
% G. Tang, X. Wang and L. Zheng

clear all
close all
rand('state',1) 
randn('state',1)

% Problem data 

% Number of measurements
n = 101;
n_h = (n-1)/2;
% The signal has 14 spectral lines and 14 spikes, but the locations and
% amplitudes are symmetric so that the signal is real for ease of
% visualization
k = 5; 
s = 5;

% Random frequency locations with a minimum separation of 2.8 / (n-1)
distance = 2.8 ./ (n - 1);
jitter_ratio = 0.5;
jitter_factor = 10;
jitter = distance/jitter_factor;
freq_sines_aux = ( (distance/2) : distance : (k+1)*distance )';
offsets = cumsum(jitter.*rand(k,1).*(rand(k,1)<jitter_ratio));
freq_sines_plot = distance + freq_sines_aux(1:k) + offsets;

% Spectral lines

% For ease of visualization, the signal is real in the time domain
freq_sines = [-flipud(freq_sines_plot); freq_sines_plot];
% Spectral-line amplitudes
random_signs = 2 * (0.5 - (rand(k,1) > 0.5) ); 
amp_sines_plot = random_signs .* rand(k,1);
% The amplitudes are symmetric so that the signal is real in the time domain
amp_sines = [flipud(amp_sines_plot); amp_sines_plot]; 
% Samples
ind = -n_h:1:n_h;
F = exp(-1i*2*pi*ind'*freq_sines'); 
y_clean = real(F*amp_sines);

% Impulsive noise

z= zeros(n,1);
aux_spikes = randperm(n_h);
ind_spikes_orig = aux_spikes(1:s);
random_signs_spikes = 2 * (0.5 - (rand(s,1) > 0.5) );
z(n_h+1+ ind_spikes_orig ) = 1.5 * max(real(F * amp_sines)) ...
                             * random_signs_spikes .* rand(s,1);
z(n_h+1- ind_spikes_orig ) = z(n_h+1+ ind_spikes_orig );
clean_ind = setdiff(1:n,ind_spikes_orig);

% Data

y = y_clean + z;

% Linear nonparametric method: Periodogram with a Gaussian window

N = 5e3;
period_values = 0.5 + (1:N)./N-1/2*ones(1,N);
alpha = 1.5;
wingaussian = gausswin(n,alpha);

period_gauss = flipud(periodogram(complex(y),wingaussian,N));
period_gauss_clean = flipud(periodogram(complex(y_clean),wingaussian,N));
aux_max = max(period_gauss);

figure
stem(freq_sines_plot,aux_max * abs(amp_sines_plot),'r')
hold on
plot(period_values(1:N/2),period_gauss_clean(1:N/2),'g')
xlim([0,0.3])
legend('Spectrum','Periodogram')
title('Periodogram (no outliers)')

figure
stem(freq_sines_plot,aux_max * abs(amp_sines_plot),'r')
hold on
plot(period_values(1:N/2),period_gauss(1:N/2))
xlim([0,0.3])
legend('Spectrum','Periodogram')
title('Periodogram (outliers in the data)')

% Parametric method: MUSIC

m = 40;
aux_cov_clean = corrmtx(y_clean,m-1,'covariance');
cov_matrix_clean = aux_cov_clean'*aux_cov_clean;
est_music_clean = change_var(rootmusic(cov_matrix_clean,k*2,'corr'));
% Estimate amplitudes
F_music_clean = exp(-1i*2*pi*ind'*est_music_clean'); 
amp_music_clean = (F_music_clean\y_clean);
aux_cov = corrmtx(y,m-1,'covariance');
cov_matrix = aux_cov'*aux_cov;
est_music = change_var(rootmusic(cov_matrix,k*2,'corr'));
% Estimate amplitudes
F_music = exp(-1i*2*pi*ind'*est_music'); 
amp_music = (F_music\y);

figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(est_music_clean,amp_music_clean,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('MUSIC (no outliers)')

figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(est_music,amp_music,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('MUSIC (outliers in the data)')

% Nonparametric method based on semidefinite programming

% Dense noise model

% We use the true l2 norm of the corruptions
sigma = norm(y-y_clean);

[f_sdp, amp_sdp] = sdp_dense_noise(y,sigma);
[f_sdp_clean, amp_sdp_clean] = sdp_no_noise(y_clean);

figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(f_sdp_clean,amp_sdp_clean,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('SDP with dense-noise model (no outliers)')

figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(f_sdp,amp_sdp,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('SDP with dense-noise model (outliers in the data)')

% Sparse noise model

lambda = 1/sqrt(n);

[f_sdp_sparse, amp_sdp_sparse, ind_spikes, amp_spikes] = ...
                        sdp_outliers(y, lambda);
[f_sdp_sparse_clean, amp_sdp_sparse_clean, ind_spikes_clean, ...
    amp_spikes_clean] = sdp_outliers(y_clean, lambda);

figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(f_sdp_sparse_clean,amp_sdp_sparse_clean,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('SDP with sparse-noise model (no outliers)')

figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(f_sdp_sparse,amp_sdp_sparse,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('SDP with sparse-noise model (outliers in the data)')

figure
stem(z,'ob')
hold on
stem(real(amp_spikes),'xr')
xlim([n_h+1 n])
legend('Spikes','Estimate')
title('SDP with sparse-noise model (outliers in the data)')

% Greedy demixing

max_iters = 100;
thresh = 0.2; % Smaller than the smallest true coefficient
[f_est_demixing, spike_est_demixing, coeffs_freq_demixing, coeffs_spikes_demixing] ...
        = greedy_demixing(y, -n_h:n_h, thresh, max_iters, false);
figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(f_est_demixing,coeffs_freq_demixing,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('Greedy demixing (outliers in the data)')

figure
stem(z,'ob')
hold on
stem(spike_est_demixing,coeffs_spikes_demixing,'xr')
xlim([n_h+1 n])
legend('Spikes','Estimate')
title('Greedy demixing (outliers in the data)')

% Greedy demixing with local optimization

max_iters = 100;
thresh = 0.1;
[f_est_demixing, spike_est_demixing, coeffs_freq_demixing, coeffs_spikes_demixing] ...
        = greedy_demixing(y, -n_h:n_h, thresh, max_iters, true);
figure
stem(freq_sines_plot, amp_sines_plot,'ob')
hold on
stem(f_est_demixing,coeffs_freq_demixing,'xr')
xlim([0,0.3])
legend('Spectrum','Estimate')
title('Greedy demixing with local optimization (outliers in the data)')

figure
stem(z,'ob')
hold on
stem(spike_est_demixing,coeffs_spikes_demixing,'xr')
xlim([n_h+1 n])
legend('Spikes','Estimate')
title('Greedy demixing with local optimization (outliers in the data)')



