% Atomic-norm denoising in the presence of outliers, based on solving 
% the problem
% 
% min_{g,z} || g ||_A / sqrt(n) + lambda || z ||_1 + gamma / 2 || y - g - z ||^2  
% 
% for lambda, lambda_0 > 0, using an equivalent SDP formulation solved
% applying ADMM.
%
% For more information see the paper "Demixing Sines and Spikes: Spectral 
% Super-resolution in the Presence of Outliers" by C. Fernandez-Granda,
% G. Tang, g. Wang and L. Zheng

function [g,z,g_log] = denoising_admm(y,lambda,gamma, maxIter)

n = length(y);

% Change of variables

lambda_0 = lambda / gamma;
xi = 1 / (gamma * sqrt(n));

% Parameters

rho = 0.1; % Penalty parameter in the augmented Lagrangian

% Variables
g = zeros(n,1);
Z_old = zeros(n+1);
Lambda = zeros(n+1);
M = 1./[n; 2*((n-1):-1:1)'];
e1 = zeros(n,1); 
e1(1)=xi/2/rho;
z_old=zeros(n,1);

for count = 1:maxIter  
    
    % Updates
    z = sign(y-g).*(abs(y-g)-lambda_0).*(abs(y-g)>lambda_0);
    g = 1/(1+2*rho)*(y-z_old + 2*rho*Z_old(1:n,n+1) + 2*Lambda(1:n,n+1));
    u = M.*(toeplitz_adjoint(Z_old(1:n,1:n) + Lambda(1:n,1:n)/rho)) - e1;
    t = Z_old(n+1,n+1) + (Lambda(n+1,n+1) - xi/2)/rho;
    z_old = z;
   
    W = [toeplitz(u),g;g',t];
    
    Q = W - Lambda/rho;
    [V,E] = eig((Q+Q')/2);   
    e = diag(E);
    idx = (e>0);
    Z = V(:,idx)*diag(e(idx))*V(:,idx)';
    Z = (Z+Z')/2;
   
    % Update the dual multiplier
    Lambda = Lambda + rho*(Z-W);
    Z_old=Z;
    
    g_log{count} = g;
    
end


function T = toeplitz_adjoint(A)
% For a square matrix A, find the Toeplitz approximation
% by averaging along the diagonals.
N = size(A,1);
T = zeros(N,1);
T(1) = sum(diag(A));
for n = 1:(N-1)
    T(n+1) = 2*sum(diag(A,n));
end