% Spectral super-resolution in the presence of dense noise.
%
% Recovery is carried out by solving the problem
%
% max_u Re(y'u) - sigma ||u||_2
% subject to max_ j |( F*u )_j | <= 1
% 
% using an equivalent SDP formulation. The script uses CVX
% (http://cvxr.com/cvx/).
%
% For more information see the paper "Super-resolution from noisy data" 
% by E. Candes and C. Fernandez-Granda. 

function [f_est, amp_est] = sdp_dense_noise(y, sigma, tol)

n = length(y);
n_h = (n-1)/2;
ind = -n_h:1:n_h;

cvx_precision best
cvx_solver sdpt3
cvx_begin sdp 
    variable X(n+1,n+1) hermitian;
    X >= 0;
    X(n+1,n+1) == 1;
    trace(X) == 2;
    for j = 1:n-1,
        sum(diag(X,j)) == X(n+1-j,n+1);
    end
    maximize(real(X(1:n,n+1)'*y)-sigma*norm(X(1:n,n+1)))
cvx_end

u = X(1:n,n+1);
aux_u =- conv(u,flipud(conj(u)));
aux_u(2*n_h+1)=1+aux_u(2*n_h+1);
roots_pol = roots(flipud(aux_u));

% Parameter for thresholding the polynomial corresponding to the dual 
% solution
tol = 1e-4; 

% Isolate roots on the unit circle
roots_detected = roots_pol(abs(1-abs(roots_pol)) < tol);
[~,ind_sort]=sort(real(roots_detected));
roots_detected = roots_detected(ind_sort);
% Roots are double so take 1 and out of 2 and compute argument
f_est = sort(angle(roots_detected(1:2:end))/2/pi);
F_est = exp(-1i*2*pi*ind'*f_est'); % estimated Fourier matrix
amp_est = F_est\y; % estimated amplitudes
