% Spectral super-resolution in the presence of outliers. Recovery is 
% carried out by solving
% 
% max_u Re(y'u) 
% subject to max_t |(F*u)(f) | <= 1, max_j |u_j| <= lambda
% 
% for a regularization parameter lambda >0, using an equivalent SDP 
% formulation. The script uses CVX (http://cvxr.com/cvx/).
%
% For more information see the paper "Demixing Sines and Spikes: Spectral 
% Super-resolution in the Presence of Outliers" by C. Fernandez-Granda,
% G. Tang, X. Wang and L. Zheng

function [f_est, amp_est, ind_spikes, amp_spikes]...
            = sdp_outliers(y, lambda)

n = length(y);
n_h = (n-1)/2;
k = -n_h:1:n_h;

cvx_precision best
cvx_solver sdpt3
cvx_begin sdp 
    variable X(n+1,n+1) hermitian;
    X >= 0;
    X(n+1,n+1) == 1;
    trace(X) == 2;
    for j = 1:n-1,
        sum(diag(X,j)) == X(n+1-j,n+1);
    end
    abs(X(1:n,n+1))<=lambda;
    maximize(real(X(1:n,n+1)'*y))
cvx_end

% Parameter for thresholding dual solution and the corresponding polynomial 
tol = 1e-4; 

u = X(1:n,n+1);
ind_spikes = find(abs((abs(u))-lambda)< tol );
clean_measurements = setdiff(1:n,ind_spikes);
% Roots of dual polynomial
aux_u =- conv(u,flipud(conj(u)));
aux_u(2*n_h+1)=1+aux_u(2*n_h+1);
roots_pol = roots(flipud(aux_u));


% Isolate roots on the unit circle
roots_detected = roots_pol(abs(1-abs(roots_pol)) < tol);
[~,ind_sort]=sort(real(roots_detected));
roots_detected = roots_detected(ind_sort);
% Roots are double so take 1 and out of 2 and compute argument
f_est = sort(angle(roots_detected(1:2:end))/2/pi);
F_est = exp(-1i*2*pi*k(clean_measurements)'*f_est'); % estimated Fourier matrix
amp_est = F_est\y(clean_measurements); % estimated amplitudes
F_spikes = exp(-1i*2*pi*k(ind_spikes)'*f_est'); 
amp_spikes =zeros(n,1);
amp_spikes(ind_spikes) = y(ind_spikes)-F_spikes*amp_est;
