# Aleksandar Donev, December 2002
# Models for jamming in hard-sphere packings
# Part I: Common declarations

param problem_name symbolic default "Packing";
param input_file symbolic default "Packing.r.dat";
param output_file symbolic default "Packing.dat";
param plot_file symbolic default "Packing.wrl";

param d integer in {2,3} default 2; # Dimensionality
param n integer >0 default 1; # Number of nodes
param m integer >0 default 1; # Number of real arcs
param m_ integer >=0 default 0; # Number of virtual (periodic) arcs
param n_ids>0 default 1; # Number of node categories
   # 1 for monodisperse, 2 for bidisperse packings, etc.

set NODES:={1..n} ordered; # All n nodes
set DIMS:={1..d} ordered; # x,y,z, etc.
set REAL_ARCS:={1..m} ordered; # Real arcs
set VIRTUAL_ARCS:={m+1..m+m_} ordered; # Virtual arcs
set ARCS:={1..m+m_} ordered; # All arcs
set STRAIN_DIMS:={1..d*(d+1)/2} ordered; # The strain space dimensionality
set IDs={1..n_ids} ordered; # Different types of spheres

param r {NODES,DIMS}; # Positions of the nodes (configuration)
   # This must be supplied in the data

param ID {NODES} in IDs default 1; # The type of each node (i.e. sphere size)

# The connectivity of graph G:
param H{ARCS} within {NODES}; # Heads
param T{ARCS} within {NODES}; # Tails

# Lattice vectors:
param L {DIMS, DIMS};

# Connectivity of graph G_ (Wrapping around torus)
param r_ {VIRTUAL_ARCS,DIMS};

# Other parameters related to tensegrity arcs:
param SCB {ARCS} in {-1,0,1} default 1;
   # Is this a strut, a cable, or a bar--IGNORED for packings!
   
param l_eq {ARCS} >=0.0 default 1.0; # Equilibrium lengths
   # Also IGNORED for packings, but used to determine length-scales

param l_eq_max >0 := max {a in ARCS} l_eq[a];
param l_eq_min >0 := min {a in ARCS} l_eq[a];

# Sometimes we perturb the positions at the start
# These are also used in iterative schemes to hold the new configuration of the packing!
param perturbation >=0 default 0;
param r_perturbed {i in NODES, k in DIMS} default r[i,k] + perturbation*l_eq_min*Uniform(-1,1);
param L_perturbed {i in DIMS, j in DIMS} default L[i,j] + perturbation*l_eq_min*Uniform(-1,1);
param r__perturbed {a in VIRTUAL_ARCS, k in DIMS} default
                   r_[a,k]+perturbation*l_eq_min*Uniform(-1,1);

# The directions along the arcs:
param l_ {a in ARCS, k in DIMS} default
   if (a in REAL_ARCS) then 
      r_perturbed[H[a],k]-r_perturbed[T[a],k]
   else
      r_perturbed[H[a],k]-r_perturbed[T[a],k]+r__perturbed[a,k]
   ;

param l {a in ARCS} default sqrt( sum {k in DIMS} (l_[a,k]^2) );
   # The lengths of the arcs

param u {a in ARCS, k in DIMS} default l_[a,k]/l[a] ; 
   # Unit vectors along the arcs

# Typical length-scales (very important!):
param l_max >0 default max {a in ARCS} l[a];
param l_min >0 default min {a in ARCS} l[a];
   # I usually take the diameter D=l_min!
param l_eps >0 default 1.0E-10;
   # Precision of data

param f_min>=0 default 1E-6; # Treshold for rounding toward zero
param I_min>=0 default 1E-6; # Also a treshold

param solver_verbosity>=0, integer default 1;

# EOF

