% Script to compare different algorithms to perform spectral 
% super-resolution. The data is of the form
% 
% y = Fx + z + w
%
% Where Fx corresponds to n samples of a multisinusoidal signal, x to the
% amplitudes of the spectrum of the signal, z to a sparse perturbation and
% w a dense perturbation.
%
% Traditional methods fail in the presence of outliers, whereas an 
% approach based on convex programming, which models the sparse
% perturbation explicitly, is capable of detecting the outliers and 
% super-resolve the spectrum of the multisinusoidal component.
%
% For more information see the paper "Demixing Sines and Spikes: Spectral 
% Super-resolution in the Presence of Outliers" by C. Fernandez-Granda,
% G. Tang, X. Wang and L. Zheng

clear all
close all
rand('state',1) 
randn('state',1)

% Problem data 

% Number of measurements
n = 101;
n_h = (n-1)/2;
% The signal has 14 spectral lines and 14 spikes, but the locations and
% amplitudes are symmetric so that the signal is real for ease of
% visualization
k = 5; 
s = 5;

% Random frequency locations with a minimum separation of 2.8 / (n-1)
distance = 2.8 ./ (n - 1);
jitter_ratio = 0.5;
jitter_factor = 10;
jitter = distance/jitter_factor;
freq_sines_aux = ( (distance/2) : distance : (k+1)*distance )';
offsets = cumsum(jitter.*rand(k,1).*(rand(k,1)<jitter_ratio));
freq_sines_plot = distance + freq_sines_aux(1:k) + offsets;

% Spectral lines

% For ease of visualization, the signal is real in the time domain
freq_sines = [-flipud(freq_sines_plot); freq_sines_plot];
% Spectral-line amplitudes
random_signs = 2 * (0.5 - (rand(k,1) > 0.5) ); 
amp_sines_plot = random_signs .* rand(k,1);
% The amplitudes are symmetric so that the signal is real in the time domain
amp_sines = [flipud(amp_sines_plot); amp_sines_plot]; 
% Samples
ind = -n_h:1:n_h;
F = exp(-1i*2*pi*ind'*freq_sines'); 
y_clean = real(F*amp_sines);

% Impulsive noise

z = zeros(n,1);
aux_spikes = randperm(n_h);
ind_spikes_orig = aux_spikes(1:s);
ind_spikes_plot = n_h + 1 + ind_spikes_orig;
random_signs_spikes = 2 * (0.5 - (rand(s,1) > 0.5) );
z(n_h+1+ ind_spikes_orig ) = 1.5 * max(real(F * amp_sines)) ...
                             * random_signs_spikes .* rand(s,1);
z(n_h+1- ind_spikes_orig ) = z(n_h+1+ ind_spikes_orig );
clean_ind = setdiff(1:n,ind_spikes_orig);

% Dense noise

% Data

y = y_clean + z;

% Dense noise variance
noise_variance_vect = [0.061 0.32];

% Thresholds for greedy demixing
thresh_vector = [0.3 2];

for ind = 1:length(noise_variance_vect)

    % Dense noise
    
    noise_variance = noise_variance_vect(ind);
    w = noise_variance * randn(n,1);
    SNR = 20*log10(norm(y_clean)/norm(w))
    
    % Data
    y = y_clean + z + w;
    
    % SDP-based method
    lambda = 1 / sqrt(n);
    sigma = 1.5 * norm(w);
    [f_sdp, amp_sdp, spikes_est] = sdp_outliers_dense_noise(y, lambda, sigma);
    % We align the estimate with the part of the data that is plotted
    spikes_est_plot = spikes_est(spikes_est > n_h+1); 
    
    figure
    stem(freq_sines_plot, amp_sines_plot,'ob')
    hold on
    stem(f_sdp,amp_sdp,'xr')
    xlim([0,0.2])
    legend('Spectrum','Estimate')
    legend('Spikes','Estimate')
    title(['SDP with sparse + dense noise model (SNR: ' ...
            num2str(SNR) ' dB)'])
    
    
    figure
    stem(abs(y),'b','Marker','none')
    hold on
    plot(ind_spikes_plot,abs(y(ind_spikes_plot)),'ok')
    plot(spikes_est,abs(y(spikes_est)),'xr')
    % We plot half of the data, since they are symmetric
    xlim([n_h+1 n])
    legend('Data','Outliers','Detected')
    title(['SDP with sparse + dense noise model (SNR: ' ...
            num2str(SNR) ' dB)'])
           
    % Greedy demixing

    max_iters = 100;
    thresh = thresh_vector(ind); 
    [f_est_demixing, spike_est_demixing, coeffs_freq_demixing, coeffs_spikes_demixing] ...
            = greedy_demixing(y, -n_h:n_h, thresh, max_iters, false);
    figure
    stem(freq_sines_plot, amp_sines_plot,'ob')
    hold on
    stem(f_est_demixing,coeffs_freq_demixing,'xr')
    xlim([0,0.3])
    legend('Spectrum','Estimate')
    title(['Greedy demixing  (SNR: ' num2str(SNR) ' dB)'])

    figure
    stem(abs(y),'b','Marker','none')
    hold on
    plot(ind_spikes_plot,abs(y(ind_spikes_plot)),'ok')
    plot(spike_est_demixing,abs(y(spike_est_demixing)),'xr')
    % We plot half of the data, since they are symmetric
    xlim([n_h+1 n])
    legend('Data','Outliers','Detected')
    title(['Greedy demixing  (SNR: ' num2str(SNR) ' dB)'])

    % Greedy demixing with local optimization

    max_iters = 100;
    thresh = thresh_vector(ind); % Smaller than the smallest true coefficient
    [f_est_demixing, spike_est_demixing, coeffs_freq_demixing, coeffs_spikes_demixing] ...
            = greedy_demixing(y, -n_h:n_h, thresh, max_iters, true);
    figure
    stem(freq_sines_plot, amp_sines_plot,'ob')
    hold on
    stem(f_est_demixing,coeffs_freq_demixing,'xr')
    xlim([0,0.3])
    legend('Spectrum','Estimate')
    title(['Greedy demixing with local optimization (SNR: ' ...
            num2str(SNR) ' dB)'])

    figure
    stem(abs(y),'b','Marker','none')
    hold on
    plot(ind_spikes_plot,abs(y(ind_spikes_plot)),'ok')
    plot(spike_est_demixing,abs(y(spike_est_demixing)),'xr')
    % We plot half of the data, since they are symmetric
    xlim([n_h+1 n])
    legend('Data','Outliers','Detected')
    title('Greedy demixing with local optimization (outliers in the data)')


end