% Script to perform atomic-norm denoising. The data is of the form
% 
% y = Fx + z + w
%
% Where Fx corresponds to n samples of a multisinusoidal signal, x to the
% amplitudes of the spectrum of the signal, z to a sparse perturbation and
% w a dense perturbation.
%
% For more information see the paper "Demixing Sines and Spikes: Spectral 
% Super-resolution in the Presence of Outliers" by C. Fernandez-Granda,
% G. Tang, X. Wang and L. Zheng

clear all
close all
rand('state',1) 
randn('state',1)

% Problem data 

% Number of measurements
n = 101;
n_h = (n-1)/2;
% The signal has 14 spectral lines and 14 spikes, but the locations and
% amplitudes are symmetric so that the signal is real for ease of
% visualization
k = 5; 
s = 5;

% Random frequency locations with a minimum separation of 2.8 / (n-1)
distance = 2.8 ./ (n - 1);
jitter_ratio = 0.5;
jitter_factor = 10;
jitter = distance/jitter_factor;
freq_sines_aux = ( (distance/2) : distance : (k+1)*distance )';
offsets = cumsum(jitter.*rand(k,1).*(rand(k,1)<jitter_ratio));
freq_sines_plot = distance + freq_sines_aux(1:k) + offsets;

% Spectral lines

% For ease of visualization, the signal is real in the time domain
freq_sines = [-flipud(freq_sines_plot); freq_sines_plot];
% Spectral-line amplitudes
random_signs = 2 * (0.5 - (rand(k,1) > 0.5) ); 
amp_sines_plot = random_signs .* rand(k,1);
% The amplitudes are symmetric so that the signal is real in the time domain
amp_sines = [flipud(amp_sines_plot); amp_sines_plot]; 
% Samples
ind = -n_h:1:n_h;
F = exp(-1i*2*pi*ind'*freq_sines'); 
y_clean = real(F*amp_sines);

% Impulsive noise

z = zeros(n,1);
aux_spikes = randperm(n_h);
ind_spikes_orig = aux_spikes(1:s);
ind_spikes_plot = n_h + 1 + ind_spikes_orig;
random_signs_spikes = 2 * (0.5 - (rand(s,1) > 0.5) );
z(n_h+1+ ind_spikes_orig ) = 1.5 * max(real(F * amp_sines)) ...
                             * random_signs_spikes .* rand(s,1);
z(n_h+1- ind_spikes_orig ) = z(n_h+1+ ind_spikes_orig );
clean_ind = setdiff(1:n,ind_spikes_orig);

% Denoising in the presence of sparse noise

% Data
y = y_clean + z;

lambda = 1 / sqrt(n);
[g_est, z_est] = sdp_denoising_outliers(y, lambda);

figure
plot(real(y),'dg');
hold on,
plot(real(y_clean),'ob');
plot(real(g_est),'--xr');
xlim([n_h+2 n])
legend('Data','Signal','Estimates')
title('Atomic-norm denoising in the presence of outliers')

% Denoising in the presence of sparse and dense noise

% Dense noise variance
noise_variance_vect = [0.061 0.32];

for ind = 1:length(noise_variance_vect)

    % Dense noise
    noise_variance = noise_variance_vect(ind);
    w = noise_variance * randn(n,1);
    SNR = 20*log10(norm(y_clean)/norm(w))
    
    % Data
    y = y_clean + z + w;
    
    lambda = 1 / sqrt(n);
    gamma = 1 / norm(w); 
    
    % SDP-based denoising, implemented with CVX
    
    g_est_cvx = sdp_denoising_outliers_dense_noise(y, lambda, gamma);
    
    mse_cvx=norm(g_est_cvx-y_clean,2)/sqrt(n)
    
    % SDP-based denoising, implemented with ADMM
    
    max_iter = 1e3;     
    [g_est_admm,z,g_log] = denoising_admm(y,lambda,gamma, max_iter);
    
    % Compute the MSE of ADMM at each iteration, check that it converges to
    % the MSE of the CVX result
    
    for iter=1:length(g_log)
        mse_admm(iter)=norm(g_log{iter}-y_clean,2)/sqrt(n);
    end

    figure
    semilogy(abs(mse_admm - mse_cvx))
    title('Convergence of the MSE of ADMM to the MSE achieved by CVX')
    xlabel('Iterations','fontsize',12)
    
    figure
    plot(real(y),'dg');
    hold on,
    plot(real(y_clean),'ob');
    plot(real(g_est_admm),'--xr');
    xlim([n_h+2 n])
    legend('Data','Signal','Estimates')
    title(['Atomic-norm denoising in the presence of outliers and dense noise (SNR: ' ...
            num2str(SNR) ' dB)'])
  
end