/*  Code for Monte Carlo class

http://www.math.nyu.edu/faculty/goodman/teaching/MonteCarlo12/ClassHomePage.php.html

The author gives permission for anyone to use this publicly posted 
code for any purpose.  The code was written for teaching, not research 
or commercial use.  It has not been tested thoroughly and probably has
serious bugs.  Results may be inaccurate, incorrect, or just wrong. */


/*  The sampler and definition of the PDF being sampled.        */

//      Take one MCMC step for the separation problem

#include "header.h"

#define ad(i,j) ((np2*(i))+(j))   /* The address in Xp of site (i,j) */


void   fSamp(       // A procedure to take one MCMC step for the mixing model
  int    X[],       // The state, an (n+2)*(n+2) lattice of zeros and ones.  Changed
  double be,        // The inverse temperature
  int    n) {       // n^2 = number of sites that can change.  Do not change boundary sites.

  int i,  j;        // One of a pair of active lattice sites
  int ip, jp;       // a neighbor of (i,j) (for "i prime" and "j prime")
  int np2 = n + 2;  // The dimensioned size of the X array, including borders

  
//       Choose an active lattice site in X at random

  i = (int) ( genrand()*n );    // the (int) caste rounds down to the nearest int.
  j = (int) ( genrand()*n );    // a number in {0,...,n-1}
  i++;                       // a number in {1,...,n} 
  j++;
  
//      Choose a neighbor of (i,j) 
  
  if ( genrand() < .5 ) {    // Toss a coin to decide whether the ...
                             // ... link is horizontal or vertical.
                             
//             The link is horizontal

    if ( genrand() < .5 ) {  // Toss a coin to decide whether to ...
                             // ... go left or right.
       ip = i+1;             // go right
       jp = j;}
    else {
       ip = i -1;            // go left
       jp = j;}
   }
  
  //             The link is vertical

  else {
    if ( genrand() < .5 ) {  // Toss a coin to decide whether to ...
                             // ... go up or down.
       ip = i;               // go up
       jp = j+1;}
    else {
       ip = i;               // go down
       jp = j-1;}
   }
  
//     Reject neighbors on the boundary

   if ( ( ip == 0 ) || ( ip == n+1 ) || ( jp == 0 ) || ( jp == n+1 ) ) return;
  
//     Do nothing if both occupied or both un-occupied

   if ( X[ad( i, j)] == X[ad( ip, jp)] ) return;
  
   int on, onp;             // the number of occupied neighbors for (i,j) and (ip,jp)
   on  = 0;
   onp = 0;
   if ( X[ad( i +1, j   )] == 1) { on++;   }
   if ( X[ad( i -1, j   )] == 1) { on++;   }
   if ( X[ad( i   , j +1)] == 1) { on++;   }
   if ( X[ad( i   , j -1)] == 1) { on++;   }
   if ( X[ad( ip+1, jp  )] == 1) { onp++;  }
   if ( X[ad( ip-1, jp  )] == 1) { onp++;  }
   if ( X[ad( ip  , jp+1)] == 1) { onp++;  }
   if ( X[ad( ip  , jp-1)] == 1) { onp++;  }
  
   double dPhi;        // the energy change for an (i,j) <--> (ip,jp) swap
   if ( X[ad( i, j)] == 1 )
      dPhi = on  - (onp - 1);   // dPhi = energy after swap - energy before swap
   else
      dPhi = onp - (on  - 1);   // it's (on-1) on includes a supposed from (i,j) to (ip,jp).
  
   if ( dPhi > 0) {                                //  move would increase energy --> Metropolis rejection step
      if ( genrand() > exp( -be*dPhi ) ) return;   // the proposed move is rejected
     }
  
   if ( X[ad( i,  j)]  == 1 ){     // The proposal is accepted, do the swap.
        X[ad( i , j )] = 0;
        X[ad( ip, jp)] = 1;}
   else {
        X[ad( i , j )] = 1;
        X[ad( ip, jp)] = 0;}
    
   return;     
    
 }






double   f(       // Calculate and return the probability of configuration X in
  int    X[],     // The state, an (n+2)*(n+2) lattice of zeros and ones.  Changed
  double be,      // The inverse temperature
  int    n) {     // n^2 = number of sites that can change.  Do not change boundary sites.

  int i,  j;        // One of a pair of active lattice sites
  int ip, jp;       // a neighbor of (i,j) (for "i prime" and "j prime")
  int np2 = n + 2;  // The dimensioned size of the X array, including borders
  
  double phi = 0.;  // the energy of configuration x
  
  
//  calculate the energy from vertical bonds, bonds from (i,j) to (i,j+1)
  
  for ( i = 1; i <= n; i++) {     // the i coordinate runs from 1 to n
     for ( j = 0; j <= n; j++) {  // the j coordinate starts at 0 to include boundary energy
        if ( ( X[ad(i,j)] == 1 ) && ( X[ad(i,j+1)] == 1 ) ) {  // If both sites are occupied
           phi += 1.;
          }
       }
    }
  
//  calculate the energy from horizontal bonds, bonds from (i,j) to (i+1,j)
  
  for ( i = 1; i <= n; i++) {     // the i coordinate runs from 0 to include boundary energy
     for ( j = 0; j <= n; j++) {  // the j coordinate starts at 1
        if ( ( X[ad(i,j)] == 1 ) && ( X[ad(i+1,j)] == 1 ) ) {  // If both sites are occupied
           phi += 1.;
          }
       }
    }
  
   return exp( -be*phi);
  
 }




void Xinit(       // Give X an initial configuration with density rho, and set boundary conditions
  int    X[],     // The state, an (n+2)*(n+2) lattice of zeros and ones.  Assumed allocated.
  double rho,     // The idensity
  int    n) {     // n^2 = number of non-boundary sites in each dimension

  int i,  j;        // One of a pair of active lattice sites
  int np2 = n + 2;  // The dimensioned size of the X array, including borders
  int np1 = n + 1;
  
  double phi = 0.;  // the energy of configuration x
  
  
//   Set all boundary occupations to zero
  
//   vertical boundaries 
  
  for ( j = 0; j < np2; j++) {
     X[ad(  0 , j)] = 0;    //  left boundary
     X[ad( np1, j)] = 0;    // right boundary
    }
    
//   horizontal boundaries 
  
  for ( i = 0; i < np2; i++) {
     X[ad( i,  0 )] = 0;    //  left boundary
     X[ad( i, np1)] = 0;    // right boundary
    }
  
//   Fill in the central part with occupations that can move

   int k = 0;               // Counts how many lattice sites you have visited so far.
   for ( i = 1; i <= n; i++){
      for ( j = 1; j <= n; j++) {
         if (k < rho*n*n)
            X[ad(i,j)] = 1;
         else
            X[ad(i,j)] = 0;
         k++;                 // count lattice sites visited
        }
     }
  
   return;
 }





void Xprint(      // Print X
  int    X[],     // The state, an (n+2)*(n+2) lattice of zeros and ones.  
  int    n) {     // n^2 = number of non-boundary sites in each dimension

  int i,  j;        // One of a pair of active lattice sites
  int np2 = n + 2;  // The dimensioned size of the X array, including borders
  
  for ( j = n+1; j >= 0; j--){  //  row 1 goes at the bottom
     for ( i = 0; i < np2; i++)
        cout << X[ad(i,j)]<< " ";
     cout << endl;
    }  
   return;
 }

int Xindex(      // Return a unique integer for each configuration
  int    X[],    // The state, an (n+2)*(n+2) lattice of zeros and ones.  
  int    n) {    // n^2 = number of non-boundary sites in each dimension
  
  if ( n > 4 ) {
     cout << " Stopping in Xindex with n = " << n << ".  The index will overflow an int" << endl;
     return 0;   // indicate failure with 0 return value (so if (Xindex) makes sense)
    }

  int i,  j;        // One of a pair of active lattice sites
  int np2 = n + 2;  // The dimensioned size of the X array, including borders
  int np1 = n + 1;
  
  int index = 0;   //  the index, to return
  int mult  = 1;   //  will be 2^k after looking at k sites
  
//   The index is in binary sum_k X_k 2^k, where k is a site.
  
  for ( j = 1; j < np1; j++){  
     for ( i = 1; i < np1; i++) {
        if ( X[ad(i,j)] == 1 )
           index += mult;   
        mult *= 2;  
       }
    }
     
   return index;
 }



string fString(){     //  Return a string describing f, for the plot label
  return "Separation problem";
 }




  
  
  