
# Numerical Methods II, Courant Institute, NYU, spring 2018
# http://www.math.nyu.edu/faculty/goodman/teaching/NumericalMethodsII2018/index.html
#  written by Jonathan Goodman (instructor)
#  see class notes Part 1 for more discussion

# Use an ODE solver to make a movie of the dynamics of a 1D nonlinear lattice

# This is the main module that calls the ODE time stepper and makes the movie

import numpy as np
import EulerForward as ODE

import matplotlib
matplotlib.use("Agg")
import matplotlib.pyplot as plt
import matplotlib.animation as manimation

import time             # for measuring the run time


#    Description of the mathematical problem

N  = 40    # number of interacting lattice points
T  = 4000.  # integrate from t=0 to t=T
L  = 1.    # length of the point chain, at rest
A  = .1   # amplitude of initial disturbance
Tf = 1.    # time between plot frames

dt = .0005   # time step for the ODE solver

dr = L/N   # distance between lattice points, at rest

d = 2*N    # number of dynamical variables

x  = np.ndarray(d)   # dynamical variables
xp = np.ndarray(d)   # another copy, for plotting

rx = np.ndarray(N)   # position variables
rp = np.ndarray(N)   # another position variable array, for plotting
rv = np.ndarray(N)   # velocity variables

#      initial positions and velocities

for k in range(N):
    r = k*dr         # rest position of lattice point k
    rx[k] = A*np.sin(2*np.pi*r/L)   # sinusoidal disturbance
    rp[k] = r                           # equilibrium positions
    rv[k] = 0.       # lattice points start at rest

#      copy into x = (rx, rv)

for k in range(N):
    x[k]   = rx[k]      # first N components
    x[k+N] = rv[k]      # next N components




#                         dynamical integration

#     set up the movie

FFMpegWriter = manimation.writers['ffmpeg']
metadata = dict(title='lattice movie', artist='Matplotlib',
                comment='Movie support!')
writer = FFMpegWriter(fps=60, metadata=metadata)

fig = plt.figure()
l, = plt.plot( rp, rx, 'ko')
plt.axis('equal')
plt.xlim(0., L)
plt.ylim(-10*A,10*A)

t    = 0.    #   time since the beginning of the simulation
tf   = 0.    #   time since the beginning of the frame
dpi = 100    #   output plot resolution, in dots per inch

startTime = time.time()       # keep track of elapsed time

with writer.saving(fig, "lattice_movie.mp4", dpi):
    while t < T:
        if t + dt <= T:     #  Take a full time step of size dt
            x = ODE.TimeStep( x, dt)
        if t+dt > T:      #  Take the last step to get exactly to time T
            dtFinal = T - t
            x = ODE.TimeStep( x, dtFinal)
        t  += dt
        tf += dt
        
#     possibly record the next frame

        if tf + dt > Tf:
            dt_frame = Tf - tf      # time until the next frame time < dt
            xp = ODE.TimeStep( x, dt_frame)   # step to the next frame time
            rx = xp[0:N]
            currentTime = time.time()
            elapsedTime = currentTime - startTime
            titleString = 't = %8.2f, dt = %8.2e, cpu time =%8.2f sec' % (t, dt, elapsedTime)
            plt.title(titleString)
            l.set_data(rp, rx)
            writer.grab_frame()
            print "t is " + str(t)
            tf = 0.

